from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.evidences import FailedEvidence, LinkEvidence, ComponentEvidence
from codeable_detectors.js.jsDetectors import detect_js_import
from codeable_detectors.pyparsing_patterns import round_braces_block, ID
from codeable_detectors.utils import update_keyword_args


class JSMongoDBLink(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["js"]

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        is_detected = False

        import_matches = detect_js_import(ctx, "mongodb")
        if not import_matches:
            return FailedEvidence("JS mongodb import not found")

        for importMatch in import_matches:
            matches.append(importMatch)
            mongo_client_var = importMatch.kwargs["importVar"]

            mongo_connect_matches = ctx.matches_pattern(Literal(mongo_client_var) + Literal(".") +
                                                        Literal("connect") + round_braces_block + Literal(";"))

            for connect_match in mongo_connect_matches:
                collection_var_matches = DetectorContext(connect_match).matches_pattern(ID + Literal("=") +
                                                                                       ID + Literal(".") + Literal(
                    "collection") + round_braces_block + Literal(";"))
                for collectionVarMatch in collection_var_matches:
                    matches.append(collectionVarMatch)
                    collection_var = collectionVarMatch.text[:collectionVarMatch.text.find("=")].strip()
                    print("collection_var var = " + str(collection_var))
                    is_detected = True

        if not is_detected:
            return FailedEvidence("mongo DB connect and collection calls not found")

        return LinkEvidence(matches).set_properties(detector_link_types="mongoWire",
                                                    detector_technology_types=["mongo"], kwargs=kwargs)


class JSMongoDBComponent(JSMongoDBLink):
    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence

        options = update_keyword_args({'guessed_name': None}, kwargs)
        name = options["guessed_name"] + " Mongo DB"
        return ComponentEvidence(evidence.matches).set_properties(detector_name=name,
                                                                  detector_component_types="mongoDB",
                                                                  detector_link_types="mongoWire",
                                                                  detector_technology_types="mongo", kwargs=kwargs)
